******************************************************
* TIME.SRC - time-of-day clock                       *
*                                                    *
* This program implements a time-of-day clock using  *
* two cascaded MC14499 serial 7-segment display      *
* drivers.  The two MC14499s drive 8 displays, as    *
* follows:                                           *
*                                                    *
*        DISPLAY  PURPOSE                            *
*        1 & 2    hours                              *
*        3 & 4    minutes                            *
*        5 & 6    seconds                            *
*        7 & 8    not used (always blank)            *
*                                                    *
* The program is interrupt-driven and returns to     *
* MON68K after the initialize sequence.  The time is *
* set using MON68K's memory modify command at the    *
* following RAM addresses:                           *
*                                                    *
*        ADDRESS  CONTENTS                           *
*        00A000   hours                              *
*        00A001   minutes                            *
*        00A002   seconds                            *
******************************************************
DUART    EQU      $00C001   ;68681 base address
OPR_SET  EQU      14*2      ;set bit reg. (clear pin)
OPR_CLR  EQU      15*2      ;clear bit reg. (set pin)
ACR      EQU      4*2       ;auxiliary control reg.
CTUR     EQU      6*2       ;counter/timer upper reg.
CTLR     EQU      7*2       ;counter/timer lower reg.
IMR      EQU      5*2       ;interrupt mask register
IVR      EQU      12*2      ;interrupt vector register
STOP     EQU      15*2      ;stop timer command
COUNT    EQU      57600     ;0.5 second timeout
TIMESTRT EQU      $F10000FF ;initial time for clock
I2VECTOR EQU      26        ;level 2 interrupt vector

         ORG      $8000
TIME     JMP      INIT      ;user program entry point
         JMP      L2ISR     ;INT2 entry point
* initialize pointers
INIT     MOVEA.L  #DUART,A0      ;A0 points to 68681
         MOVEA.L  #DIGITS,A1     ;A1 points to digits
* initialize MC14499 and BCD codes
         MOVE.B   #7,OPR_CLR(A0) ;MC14499 inputs high
         MOVE.L   #TIMESTRT,(A1) ;init BCD digits
         MOVE.B   #$FF,4(A1)     ;init decimal points
         MOVE.B   #2,5(A1)       ;init timeout counter
* initialize interrupts
         MOVE.B   #$08,IMR(A0)
         MOVE.B   #I2VECTOR,IVR(A0)
         MOVE.W   #COUNT,D0
         MOVEP.W  D0,CTUR(A0)
         MOVE.B   #$70,ACR(A0)
         ORI.W    #$0700,SR      ;mask all interrupts
         ANDI.W   #$F9FF,SR      ;mask level 1 only
         TRAP     #14            ;return to MON68K

******************************************************
* L2ISR - Level-2 Interrupt Service Routine          *
*                                                    *
*        - clear interrupt source                    *
*        - increment time                            *
*        - output time                               *
******************************************************
L2ISR    MOVEM.L  A0-A2,-(SP)
         MOVEA.L  #DUART,A0      ;A0 -> DUART
         MOVEA.L  #DIGITS,A1     ;A1 -> BCD digits
         TST.B    STOP(A0)       ;clr interrupt source
         SUBQ.B   #1,5(A1)       ;second time here?
         BNE.S    SKIP           ;no:  do nothing
         MOVE.B   #2,5(A1)       ;yes: re-init count &
         BSR      UPDATE2        ;     update display
         MOVEA.L  #DIGITS+3,A1   ;A1 -> past digits
         MOVEA.L  #INC+1,A2      ;A2 -> past increment
         MOVE.W   #$04,CCR       ;clear X, set Z
         ABCD     -(A2),-(A1)    ;increment SECONDS
         CMP.B    #$60,(A1)      ;SECONDS > 59?
         BLT.S    SKIP           ;no:  done
         CLR.B    (A1)           ;yes: reset to 0
         LEA      1(A2),A2       ;
         ANDI.B   #$EF,CCR       ;clear X in CCR
         ABCD     -(A2),-(A1)    ;increment MINUTES
         CMP.B    #$60,(A1)      ;MINUTES > 59?
         BLT.S    SKIP           ;no:  done
         CLR.B    (A1)           ;yes: reset to 0
         LEA      1(A2),A2       ;
         ADD.B    #1,-(A1)       ;increment HOURS
         CMP.B    #$FA,(A1)      ;HOURS > 12?
         BNE.S    SKIP2          ;no:  done
         MOVE.B   #$10,(A1)
         BRA.S    SKIP
SKIP2    CMP.B    #$13,(A1)
         BNE.S    SKIP
         MOVE.B   #$F1,(A1)      ;yes: reset to 1
SKIP     MOVEM.L  (SP)+,A0-A2
         RTE

******************************************************
* UPDATE2 - send 8 BCD digits to two MC14499s        *
*                                                    *
* Clear OP0 (MC14499 ENABLE), send the decimal       *
* points and the four digits, then set OP0.  Repeat. *
* Note: The most significant digit (bits 12-15) is   *
* sent first.                                        *
*                                                    *
*        ENTER - A0 points to 68681                  *
*              - A1 points to digits                 *
*              - 4(A1) points to decimal points      *
*        EXIT  - display updated, all reg. intact    *
*        USES  - OUT4                                *
******************************************************
UPDATE2  MOVEM.W  D0-D5/A2,-(SP)
         LEA      4(A1),A2       ;A2 points to DPs
         MOVE.W   #4,D5          ;2xD5 counts MC14499s
LOOP3    MOVE.B   #1,OPR_SET(A0) ;assert ENABLE line
         MOVE.B   (A2),D0        ;get DPs in D0[0:3]
         BSR.S    OUT4           ;send them
         SUBQ.W   #2,D5
         MOVE.W   0(A1,D5.W),D0  ;get digits
         MOVE.B   #4,D3          ;D3 counts digits
LOOP4    ROL.W    #4,D0          ;align bits (0-3 1st)
         BSR.S    OUT4           ;send them
         SUBQ     #1,D3          ;last digit?
         BNE      LOOP4          ;no:  send again
         TST.W    D5             ;last MC14499?
         BNE      LOOP3          ;no:  send again
         MOVE.B   #1,OPR_CLR(A0) ;yes: done
         MOVEM.W  (SP)+,D0-D5/A2
         RTS

******************************************************
* OUT4 - send 4 bits to the MC14499.                 *
*                                                    *
* Place data bit on OP2 (MC14499 DATA), then toggle  *
* OP1 (MC14499 CLOCK).  Repeat four times.  Bit 3 is *
* sent first.  Serial data is output on OP2.         *
*                                                    *
*        ENTER:   D0[0:3] containg bits to output    *
*        EXIT:    data bits sent, all reg. intact    *
*        USES:    no suboutines                      *
******************************************************
OUT4     MOVEM.W  D0-D4,-(SP)
         ROR.B    #2,D0          ;align as D0[1,0,7,6]
         MOVE.B   #4,D3          ;use D3 as counter
LOOP2    ROL.B    #1,D0          ;put bit in D0 bit 2
         MOVE.B   D0,D2          ;save data
         ANDI.B   #$04,D0        ;mask other bits
         MOVE.B   D0,OPR_CLR(A0) ;if bit set, set pin
         EORI.B   #$04,D0        ;complement data bit
         MOVE.B   D0,OPR_SET(A0) ;if bit clr, clr pin
         MOVE.B   #$02,D4        ;create CLK pulse
         MOVE.B   D4,OPR_SET(A0) ;clear CLK pin
         NOP                     ;stretch (MC14499
         NOP                     ; needs 2 us pulse)
         MOVE.B   D4,OPR_CLR(A0) ;set CLK pin
         MOVE.B   D2,D0          ;restore data
         SUBQ     #1,D3          ;last of 4 bits?
         BNE      LOOP2          ;no:  do again
         MOVEM.W  (SP)+,D0-D4    ;yes: done
         RTS

INC      DC.B     1             ;increment for BCD add

         ORG      $A000         ;data segment (digits)
DIGITS   DS.B     4             ;4 bytes, 8 digits
DP       DS.B     1             ;decimal points
TIMEOUT  DS.B     1             ;counter for timeouts
         END      TIME
